
// FRAGPLAY.H
// Header file for producing/playing sound on-the-fly
//
// Version 1.0
// Started by Pieter Geelen, August 1997
// Copyright (c) 1997 by Palmtop BV - The Netherlands

/*
	This version can play sounds in fragments. You have a
	sound buffer of SAMPLES_PER_FRAGMENT samples. 
	You fill it, and call PlayFragment() when it is full.

    The routines should work even if you produce samples
	slower or faster than they are actually played.
*/

#ifndef _FRAGPLAY_H
#define _FRAGPLAY_H

#include <D32SND.H>
#include "fragplay.par"

#undef ASSERT
#define ASSERT(condition) // define for debugging...

// a useful constant
#define STATES_PER_FRAGMENT	STATES_PER_SAMPLE*SAMPLES_PER_FRAGMENT

// ALAW defines
#define KAlawOn	 42
#define KAlawOff 170


class TFragmentPlayer
{
public:

	TFragmentPlayer();	
	~TFragmentPlayer();

public:

	// Open the iDevSound device, return 0 or error 
	TInt Open( TSoundVolume aVolume=EVolumeSoft );
  TInt ChangeVolume( TSoundVolume aVolume );
	// Close the iDevSound device (can't fail)
	void Close();

public:
	// play an alaw fragment
	TInt PlayFragment( TInt aSamples=SAMPLES_PER_FRAGMENT );
	// wait until completely finished playing
	inline void FinishPlaying()
		{ while ( iDevSound.BytesPlayed() < iNrBytesBeingPlayed ); }
	// abort what is currently playing (if anything)
	void AbortPlaying();

  // play a tone
  void PlayTone(TInt aFrequency,TInt aDurationInMicroSeconds);
public:
	// returns nonzero if there are sound was set using Add-routines
	inline TInt SamplesWereAdded() const { return iSoundStart; }
	
  TBool SoundCapable();
public:
	// directly write samples
	inline void SetSamplesSilent() 
		{
			Mem::Fill(iAlawBuf,SAMPLES_PER_FRAGMENT,KAlawOff); 
			iSoundStart=0; 
		}
	inline void SetSample( TInt aSample, TUint8 aAlawValue ) 
		{ 
			ASSERT( aSample>=0 && aSample<SAMPLES_PER_FRAGMENT );
			iAlawBuf[aSample]=aAlawValue; 
		}
	inline void SetSamples( TInt aSample, TChar aAlawValue, TInt aLength ) 
		{
			ASSERT( aSample>=0 && aSample<SAMPLES_PER_FRAGMENT );
			ASSERT( aLength>=0 && aSample+aLength<=SAMPLES_PER_FRAGMENT );
			if ( aLength>0 )
				Mem::Fill( &iAlawBuf[aSample], aLength, aAlawValue ); 
		}
	inline void SetSamples( TInt aSample, TUint8* aAlawValues, TInt aLength ) 
		{ 
			ASSERT( aSample>=0 && aSample<SAMPLES_PER_FRAGMENT );
			ASSERT( aLength>=0 && aSample+aLength<=SAMPLES_PER_FRAGMENT );
			if ( aLength>0 )
				Mem::Copy( &iAlawBuf[aSample], aAlawValues, aLength ); 
		}
	inline void SetSamples( TInt aSample, const TDesC& aDes )
		{ 
			ASSERT( aSample>=0 && aSample<SAMPLES_PER_FRAGMENT );
			ASSERT( aDes.Length()>=0 && aSample+aDes.Length()<=SAMPLES_PER_FRAGMENT );
			Mem::Copy( &iAlawBuf[aSample], aDes.Ptr(), aDes.Length() ); 
		}

#ifdef GENERATE_BY_STATE
public:
	inline void AddState( TInt aState, TInt aValue )
	{
		ASSERT( aState >= 0 );
		ASSERT( (aState/STATES_PER_SAMPLE) <  SAMPLES_PER_FRAGMENT );
		ASSERT( (aState/STATES_PER_SAMPLE) >= iSoundStart );

		if ( iCurrentValue != aValue ) // sound change?
		{
			iCurrentValue = aValue;
			if ( aValue ) 
			{				
				// soundstart at aSample
				iSoundStart = aState/STATES_PER_SAMPLE;
			}
			else 
			{
				// soundstop at aSample
				SetSamples( iSoundStart, KAlawOn, (aState/STATES_PER_SAMPLE)-iSoundStart );
			}
		}
	}
#else
public:	
	inline void AddSample( TInt aSample, TInt aValue )
	{
		ASSERT( aSample>=0 && aSample<SAMPLES_PER_FRAGMENT );
		ASSERT( aSample>=iSoundStart );

		if ( iCurrentValue != aValue ) // sound change?
		{
			iCurrentValue = aValue;
			if ( aValue ) 
			{
				// soundstart at aSample
				iSoundStart = aSample;
			}
			else 
			{
				// soundstop at aSample
				SetSamples( iSoundStart, KAlawOn, aSample-iSoundStart );
			}
		}
	}
#endif



private:
	TSoundConfig config;
	RDevSound iDevSound;
	TInt iNrBytesBeingPlayed;
private:
	TUint8	iAlawBuf[ SAMPLES_PER_FRAGMENT ];
private:
	TInt	iSoundStart;	// sample where sound started...
	TInt	iCurrentValue;	// current sound value
};

#endif

